<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once __DIR__ . '/db.php';

function require_login() {
    if (!isset($_SESSION['admin_id'])) {
        header('Location: /admin-login.php');
        exit;
    }
}

function login($username, $password) {
    global $pdo;
    
    // Debug logging
    error_log("Login attempt for username: " . $username);
    
    if (empty($username) || empty($password)) {
        error_log("Empty username or password");
        return false;
    }
    
    try {
        $stmt = $pdo->prepare("SELECT id, username, password_hash FROM admin_users WHERE username = ?");
        $stmt->execute([$username]);
        $user = $stmt->fetch();
        
        error_log("User found: " . ($user ? 'yes' : 'no'));
        
        if ($user) {
            error_log("Stored hash: " . substr($user['password_hash'], 0, 20) . "...");
            $verify_result = password_verify($password, $user['password_hash']);
            error_log("Password verify result: " . ($verify_result ? 'true' : 'false'));
            
            if ($verify_result) {
                // Regenerate session ID for security
                session_regenerate_id(true);
                $_SESSION['admin_id'] = $user['id'];
                $_SESSION['admin_username'] = $user['username'];
                $_SESSION['login_time'] = time();
                error_log("Login successful for user: " . $username);
                return true;
            }
        }
        
    } catch (PDOException $e) {
        error_log("Database error in login: " . $e->getMessage());
        return false;
    }
    
    error_log("Login failed for user: " . $username);
    return false;
}

function logout() {
    $_SESSION = array();
    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params["path"], $params["domain"],
            $params["secure"], $params["httponly"]
        );
    }
    session_destroy();
    header('Location: /admin-login.php');
    exit;
}

function generate_csrf_token() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function verify_csrf_token($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

function csrf_field() {
    $token = generate_csrf_token();
    return "<input type='hidden' name='csrf_token' value='$token'>";
}

