document.addEventListener('DOMContentLoaded', function() {
    // Mobile Navigation Toggle
    const navToggle = document.getElementById('navToggle');
    const navMenu = document.getElementById('navMenu');
    
    if (navToggle && navMenu) {
        navToggle.addEventListener('click', function() {
            navMenu.classList.toggle('active');
        });
    }

    // Track Shipment Functionality
    const trackForm = document.getElementById('trackForm');
    const trackingResults = document.getElementById('trackingResults');
    
    if (trackForm) {
        trackForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const trackingNumber = document.getElementById('trackingNumber').value.trim();
            if (!trackingNumber) {
                showAlert('Please enter a tracking number', 'danger');
                return;
            }
            
            showLoading(true);
            
            try {
                const response = await fetch(`/api/track_lookup.php?tracking=${encodeURIComponent(trackingNumber)}`);
                const data = await response.json();
                
                if (data.found) {
                    displayTrackingResults(data.data);
                } else {
                    showNotFound();
                }
            } catch (error) {
                showAlert('Error looking up shipment. Please try again.', 'danger');
            } finally {
                showLoading(false);
            }
        });
    }

    // Form Validation
    const forms = document.querySelectorAll('form[data-validate]');
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            if (!validateForm(this)) {
                e.preventDefault();
            }
        });
    });
});

function displayTrackingResults(shipment) {
    const trackingResults = document.getElementById('trackingResults');
    if (!trackingResults) return;
    
    const statusSteps = ['Pending', 'In Transit', 'Arrived at Port', 'Cleared', 'Out for Delivery', 'Delivered'];
    const currentStatusIndex = statusSteps.indexOf(shipment.status);
    
    // Calculate progress percentage
    const progressPercent = currentStatusIndex >= 0 ? (currentStatusIndex / (statusSteps.length - 1)) * 100 : 0;
    
    trackingResults.innerHTML = `
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">Shipment Details</h3>
            </div>
            
            <div class="grid-cols-2 mb-3">
                <div>
                    <strong>Tracking Number:</strong><br>
                    ${escapeHtml(shipment.tracking_number)}
                </div>
                <div>
                    <strong>Customer:</strong><br>
                    ${escapeHtml(shipment.customer_name)}
                </div>
                <div>
                    <strong>Cargo Type:</strong><br>
                    ${escapeHtml(shipment.cargo_type)}
                </div>
                <div>
                    <strong>Status:</strong><br>
                    <span class="status-badge status-${shipment.status.toLowerCase().replace(/\s+/g, '')}">${escapeHtml(shipment.status)}</span>
                </div>
                <div>
                    <strong>Origin:</strong><br>
                    ${escapeHtml(shipment.origin)}
                </div>
                <div>
                    <strong>Destination:</strong><br>
                    ${escapeHtml(shipment.destination)}
                </div>
                <div>
                    <strong>ETA:</strong><br>
                    ${shipment.eta ? new Date(shipment.eta).toLocaleDateString() : 'TBD'}
                </div>
                <div>
                    <strong>Last Updated:</strong><br>
                    ${new Date(shipment.last_updated).toLocaleString()}
                </div>
            </div>
            
            <div class="mb-3">
                <strong>Description:</strong><br>
                ${escapeHtml(shipment.description)}
            </div>
            
            ${shipment.shipping_cost ? `
                <div class="mb-3">
                    <strong>Shipping Cost:</strong><br>
                    ₦${Number(shipment.shipping_cost).toLocaleString()}
                </div>
            ` : ''}
            
            <div class="progress-container">
                <h4 style="margin-bottom: 1rem;">Shipment Progress</h4>
                <div class="progress-bar">
                    <div class="progress-line">
                        <div class="progress-line-fill" style="width: ${progressPercent}%"></div>
                    </div>
                    ${statusSteps.map((step, index) => `
                        <div class="progress-step ${index < currentStatusIndex ? 'completed' : ''} ${index === currentStatusIndex ? 'current' : ''}">
                            <div class="progress-circle">${index + 1}</div>
                            <span class="progress-label">${step}</span>
                        </div>
                    `).join('')}
                </div>
            </div>
        </div>
    `;
    
    trackingResults.scrollIntoView({ behavior: 'smooth' });
}

function showNotFound() {
    const trackingResults = document.getElementById('trackingResults');
    if (!trackingResults) return;
    
    trackingResults.innerHTML = `
        <div class="card text-center">
            <h3 style="color: var(--danger-red); margin-bottom: 1rem;">Shipment Not Found</h3>
            <p>We couldn't find a shipment with that tracking number. Please check the number and try again.</p>
            <p style="margin-top: 1rem;">
                <a href="/get-quote.php" class="btn btn-primary">Get a New Quote</a>
            </p>
        </div>
    `;
}

function showLoading(show) {
    const form = document.getElementById('trackForm');
    const button = form?.querySelector('button[type="submit"]');
    
    if (show) {
        if (button) button.textContent = 'Searching...';
        if (form) form.classList.add('loading');
    } else {
        if (button) button.textContent = 'Track Shipment';
        if (form) form.classList.remove('loading');
    }
}

function showAlert(message, type = 'info') {
    // Remove existing alerts
    const existingAlerts = document.querySelectorAll('.alert');
    existingAlerts.forEach(alert => alert.remove());
    
    const alert = document.createElement('div');
    alert.className = `alert alert-${type}`;
    alert.textContent = message;
    
    const container = document.querySelector('.container') || document.body;
    container.insertBefore(alert, container.firstChild);
    
    // Auto-remove after 5 seconds
    setTimeout(() => {
        alert.remove();
    }, 5000);
}

function validateForm(form) {
    let isValid = true;
    const requiredFields = form.querySelectorAll('[required]');
    
    requiredFields.forEach(field => {
        if (!field.value.trim()) {
            field.style.borderColor = 'var(--danger-red)';
            isValid = false;
        } else {
            field.style.borderColor = 'var(--border-gray)';
        }
    });
    
    // Email validation
    const emailFields = form.querySelectorAll('input[type="email"]');
    emailFields.forEach(field => {
        if (field.value && !isValidEmail(field.value)) {
            field.style.borderColor = 'var(--danger-red)';
            isValid = false;
        }
    });
    
    return isValid;
}

function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

function escapeHtml(unsafe) {
    return unsafe
        .replace(/&/g, "&amp;")
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/"/g, "&quot;")
        .replace(/'/g, "&#039;");
}

function confirmDelete(message = 'Are you sure you want to delete this item?') {
    return confirm(message);
}

// Auto-hide alerts
setTimeout(() => {
    const alerts = document.querySelectorAll('.alert');
    alerts.forEach(alert => {
        alert.style.transition = 'opacity 0.3s ease';
        alert.style.opacity = '0';
        setTimeout(() => alert.remove(), 300);
    });
}, 5000);