<?php
require_once '../includes/db.php';
require_once '../includes/auth.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

// CSRF Protection
if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
    http_response_code(403);
    echo json_encode(['error' => 'Invalid request']);
    exit;
}

// Validation
$required_fields = ['name', 'phone', 'email', 'cargo_type', 'origin', 'destination', 'description'];
$data = [];

foreach ($required_fields as $field) {
    $value = trim($_POST[$field] ?? '');
    if (empty($value)) {
        http_response_code(400);
        echo json_encode(['error' => "Field '$field' is required"]);
        exit;
    }
    $data[$field] = $value;
}

// Email validation
if (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid email address']);
    exit;
}

// Cargo type validation
if (!in_array($data['cargo_type'], ['Air', 'Sea'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid cargo type']);
    exit;
}

try {
    // Generate reference code
    $year = date('Y');
    $stmt = $pdo->prepare("SELECT COUNT(*) + 1 as next_num FROM quotes WHERE ref_code LIKE ?");
    $stmt->execute(["Q-$year-%"]);
    $next_num = $stmt->fetch()['next_num'];
    $ref_code = sprintf("Q-%s-%04d", $year, $next_num);
    
    // Insert quote
    $stmt = $pdo->prepare("
        INSERT INTO quotes (ref_code, name, phone, email, cargo_type, origin, destination, description) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $ref_code,
        $data['name'],
        $data['phone'],
        $data['email'],
        $data['cargo_type'],
        $data['origin'],
        $data['destination'],
        $data['description']
    ]);
    
    // Set success message and redirect
    $_SESSION['quote_success'] = "Quote received successfully! Reference: $ref_code. We'll contact you shortly.";
    header('Location: /get-quote.php');
    exit;
    
} catch (PDOException $e) {
    error_log("Quote submission error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Failed to submit quote. Please try again.']);
}