<?php
require_once '../includes/db.php';
require_once '../includes/auth.php';
require_login();

$shipment_id = $_GET['id'] ?? null;
$from_quote_id = $_GET['from_quote'] ?? null;
$is_edit = !empty($shipment_id);

$shipment = null;
$quote = null;

// Load existing shipment for editing
if ($is_edit) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM shipments WHERE id = ?");
        $stmt->execute([$shipment_id]);
        $shipment = $stmt->fetch();
        
        if (!$shipment) {
            header('Location: /admin/shipments.php');
            exit;
        }
    } catch (PDOException $e) {
        error_log("Load shipment error: " . $e->getMessage());
        header('Location: /admin/shipments.php');
        exit;
    }
}

// Load quote data for conversion
if ($from_quote_id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM quotes WHERE id = ?");
        $stmt->execute([$from_quote_id]);
        $quote = $stmt->fetch();
    } catch (PDOException $e) {
        error_log("Load quote error: " . $e->getMessage());
    }
}

// Handle form submission
if ($_POST) {
    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        $error_message = "Invalid request. Please try again.";
    } else {
        // Validation
        $required_fields = ['customer_name', 'customer_email', 'cargo_type', 'origin', 'destination', 'description', 'status'];
        $data = [];
        $validation_errors = [];
        
        foreach ($required_fields as $field) {
            $value = trim($_POST[$field] ?? '');
            if (empty($value)) {
                $validation_errors[] = "Field '$field' is required";
            }
            $data[$field] = $value;
        }
        
        // Optional fields
        $data['eta'] = $_POST['eta'] ?? null;
        $data['shipping_cost'] = !empty($_POST['shipping_cost']) ? floatval($_POST['shipping_cost']) : null;
        
        // Validate email
        if (!filter_var($data['customer_email'], FILTER_VALIDATE_EMAIL)) {
            $validation_errors[] = "Invalid email address";
        }
        
        // Validate cargo type and status
        if (!in_array($data['cargo_type'], ['Air', 'Sea'])) {
            $validation_errors[] = "Invalid cargo type";
        }
        
        $valid_statuses = ['Pending', 'In Transit', 'Arrived at Port', 'Cleared', 'Out for Delivery', 'Delivered'];
        if (!in_array($data['status'], $valid_statuses)) {
            $validation_errors[] = "Invalid status";
        }
        
        if (empty($validation_errors)) {
            try {
                if ($is_edit) {
                    // Update existing shipment
                    $stmt = $pdo->prepare("
                        UPDATE shipments 
                        SET customer_name = ?, customer_email = ?, cargo_type = ?, 
                            origin = ?, destination = ?, description = ?, status = ?, 
                            eta = ?, shipping_cost = ?, last_updated = CURRENT_TIMESTAMP
                        WHERE id = ?
                    ");
                    
                    $stmt->execute([
                        $data['customer_name'], $data['customer_email'], $data['cargo_type'],
                        $data['origin'], $data['destination'], $data['description'],
                        $data['status'], $data['eta'], $data['shipping_cost'], $shipment_id
                    ]);
                    
                    $_SESSION['success_message'] = "Shipment updated successfully!";
                } else {
                    // Generate unique tracking number for new shipment
                    $year = date('Y');
                    $tracking_number = null;
                    $max_attempts = 10;
                    
                    for ($attempt = 1; $attempt <= $max_attempts; $attempt++) {
                        // Get the highest existing number for this year
                        $stmt = $pdo->prepare("
                            SELECT MAX(CAST(SUBSTRING(tracking_number, -3) AS UNSIGNED)) as max_num 
                            FROM shipments 
                            WHERE tracking_number LIKE ?
                        ");
                        $stmt->execute(["FTL$year-%"]);
                        $result = $stmt->fetch();
                        $next_num = ($result['max_num'] ?? 0) + 1;
                        
                        $tracking_number = sprintf("FTL%s-%03d", $year, $next_num);
                        
                        // Check if this tracking number already exists
                        $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM shipments WHERE tracking_number = ?");
                        $stmt->execute([$tracking_number]);
                        
                        if ($stmt->fetch()['count'] == 0) {
                            break; // Found unique tracking number
                        }
                        
                        if ($attempt == $max_attempts) {
                            throw new Exception("Unable to generate unique tracking number after $max_attempts attempts");
                        }
                    }
                    
                    // Insert new shipment
                    $stmt = $pdo->prepare("
                        INSERT INTO shipments (tracking_number, customer_name, customer_email, cargo_type, 
                                             origin, destination, description, status, eta, shipping_cost) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    
                    $stmt->execute([
                        $tracking_number, $data['customer_name'], $data['customer_email'], 
                        $data['cargo_type'], $data['origin'], $data['destination'], 
                        $data['description'], $data['status'], $data['eta'], $data['shipping_cost']
                    ]);
                    
                    $_SESSION['success_message'] = "Shipment created successfully! Tracking Number: $tracking_number";
                }
                
                header('Location: /admin/shipments.php');
                exit;
                
            } catch (PDOException $e) {
                error_log("Shipment save error: " . $e->getMessage());
                $error_message = "Database error: " . $e->getMessage() . " Please check your input and try again.";
            } catch (Exception $e) {
                error_log("Shipment save error: " . $e->getMessage());
                $error_message = "Error: " . $e->getMessage();
            }
        } else {
            $error_message = implode(', ', $validation_errors);
        }
    }
}

$page_title = $is_edit ? "Edit Shipment" : "Create Shipment";
require_once '../includes/header.php';
?>

<div class="container" style="max-width: 800px; margin: 0 auto;">
    <div class="flex justify-between items-center mb-3">
        <h1><?= $is_edit ? 'Edit' : 'Create' ?> Shipment</h1>
        <a href="/admin/shipments.php" class="btn btn-secondary">← Back to Shipments</a>
    </div>
    
    <?php if (isset($_SESSION['success_message'])): ?>
        <div class="alert alert-success">
            <?= htmlspecialchars($_SESSION['success_message']) ?>
        </div>
        <?php unset($_SESSION['success_message']); ?>
    <?php endif; ?>
    
    <?php if (isset($error_message)): ?>
        <div class="alert alert-danger">
            <?= htmlspecialchars($error_message) ?>
        </div>
    <?php endif; ?>
    
    <div class="card">
        <form method="POST" data-validate>
            <?= csrf_field() ?>
            
            <div class="grid-cols-2">
                <div class="form-group">
                    <label for="customer_name" class="form-label">Customer Name *</label>
                    <input type="text" id="customer_name" name="customer_name" class="form-control" 
                           value="<?= htmlspecialchars($quote['name'] ?? $shipment['customer_name'] ?? '') ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="customer_email" class="form-label">Customer Email *</label>
                    <input type="email" id="customer_email" name="customer_email" class="form-control" 
                           value="<?= htmlspecialchars($quote['email'] ?? $shipment['customer_email'] ?? '') ?>" required>
                </div>
            </div>
            
            <div class="grid-cols-2">
                <div class="form-group">
                    <label for="cargo_type" class="form-label">Cargo Type *</label>
                    <select id="cargo_type" name="cargo_type" class="form-control form-select" required>
                        <option value="">Select Type</option>
                        <?php 
                        $selected_cargo = $quote['cargo_type'] ?? $shipment['cargo_type'] ?? '';
                        foreach (['Air', 'Sea'] as $type): 
                        ?>
                            <option value="<?= $type ?>" <?= $selected_cargo === $type ? 'selected' : '' ?>>
                                <?= $type ?> Freight
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="status" class="form-label">Status *</label>
                    <select id="status" name="status" class="form-control form-select" required>
                        <?php 
                        $current_status = $shipment['status'] ?? 'Pending';
                        $statuses = ['Pending', 'In Transit', 'Arrived at Port', 'Cleared', 'Out for Delivery', 'Delivered'];
                        foreach ($statuses as $status): 
                        ?>
                            <option value="<?= $status ?>" <?= $current_status === $status ? 'selected' : '' ?>>
                                <?= htmlspecialchars($status) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
            
            <div class="grid-cols-2">
                <div class="form-group">
                    <label for="origin" class="form-label">Origin *</label>
                    <input type="text" id="origin" name="origin" class="form-control" 
                           value="<?= htmlspecialchars($quote['origin'] ?? $shipment['origin'] ?? '') ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="destination" class="form-label">Destination *</label>
                    <input type="text" id="destination" name="destination" class="form-control" 
                           value="<?= htmlspecialchars($quote['destination'] ?? $shipment['destination'] ?? '') ?>" required>
                </div>
            </div>
            
            <div class="form-group">
                <label for="description" class="form-label">Description of Goods *</label>
                <textarea id="description" name="description" class="form-control" required><?= htmlspecialchars($quote['description'] ?? $shipment['description'] ?? '') ?></textarea>
            </div>
            
            <div class="grid-cols-2">
                <div class="form-group">
                    <label for="eta" class="form-label">Estimated Arrival</label>
                    <input type="date" id="eta" name="eta" class="form-control" 
                           value="<?= $shipment['eta'] ?? '' ?>">
                </div>
                
                <div class="form-group">
                    <label for="shipping_cost" class="form-label">Shipping Cost (₦)</label>
                    <input type="number" id="shipping_cost" name="shipping_cost" class="form-control" 
                           step="0.01" min="0" value="<?= $shipment['shipping_cost'] ?? '' ?>">
                </div>
            </div>
            
            <div style="display: flex; gap: 1rem; justify-content: flex-end;">
                <a href="/admin/shipments.php" class="btn btn-secondary">Cancel</a>
                <button type="submit" class="btn btn-primary">
                    <?= $is_edit ? 'Update' : 'Create' ?> Shipment
                </button>
            </div>
        </form>
    </div>
</div>

<?php require_once '../includes/footer.php'; ?>