<?php
require_once '../includes/db.php';
require_once '../includes/auth.php';
require_login();

$search = trim($_GET['search'] ?? '');
$page = max(1, intval($_GET['page'] ?? 1));
$per_page = 20;
$offset = ($page - 1) * $per_page;

try {
    $where_clause = '';
    $params = [];
    
    if ($search) {
        $where_clause = "WHERE name LIKE ? OR email LIKE ? OR ref_code LIKE ?";
        $params = ["%$search%", "%$search%", "%$search%"];
    }
    
    // Get total count
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM quotes $where_clause");
    $stmt->execute($params);
    $total_quotes = $stmt->fetch()['total'];
    
    // Get quotes
    $stmt = $pdo->prepare("
        SELECT * FROM quotes 
        $where_clause 
        ORDER BY created_at DESC 
        LIMIT $per_page OFFSET $offset
    ");
    $stmt->execute($params);
    $quotes = $stmt->fetchAll();
    
} catch (PDOException $e) {
    error_log("Quotes page error: " . $e->getMessage());
    $quotes = [];
    $total_quotes = 0;
}

$page_title = "Manage Quotes";
require_once '../includes/header.php';
?>

<div class="container">
    <div class="flex justify-between items-center mb-3">
        <h1>Manage Quotes</h1>
        <a href="/admin/dashboard.php" class="btn btn-secondary">← Back to Dashboard</a>
    </div>
    
    <div class="card">
        <div class="search-container">
            <form method="GET" class="flex gap-2">
                <input 
                    type="text" 
                    name="search" 
                    value="<?= htmlspecialchars($search) ?>"
                    placeholder="Search by name, email, or reference..."
                    class="form-control search-input"
                >
                <button type="submit" class="btn btn-primary">Search</button>
                <?php if ($search): ?>
                    <a href="/admin/quotes.php" class="btn btn-secondary">Clear</a>
                <?php endif; ?>
            </form>
        </div>
        
        <?php if (empty($quotes)): ?>
            <div class="text-center p-2">
                <p style="color: var(--medium-gray);">
                    <?= $search ? 'No quotes found matching your search.' : 'No quotes submitted yet.' ?>
                </p>
            </div>
        <?php else: ?>
            <div class="table-container">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Reference</th>
                            <th>Name</th>
                            <th>Email</th>
                            <th>Type</th>
                            <th>Route</th>
                            <th>Date</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($quotes as $quote): ?>
                            <tr>
                                <td><?= htmlspecialchars($quote['ref_code']) ?></td>
                                <td><?= htmlspecialchars($quote['name']) ?></td>
                                <td><?= htmlspecialchars($quote['email']) ?></td>
                                <td><?= htmlspecialchars($quote['cargo_type']) ?></td>
                                <td><?= htmlspecialchars($quote['origin']) ?> → <?= htmlspecialchars($quote['destination']) ?></td>
                                <td><?= date('M j, Y', strtotime($quote['created_at'])) ?></td>
                                <td>
                                    <div style="display: flex; gap: 0.5rem;">
                                        <a href="/admin/email_send.php?quote_id=<?= $quote['id'] ?>" class="btn btn-small btn-secondary">Email</a>
                                        <a href="/admin/shipment_form.php?from_quote=<?= $quote['id'] ?>" class="btn btn-small btn-primary">Convert</a>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <?php if ($total_quotes > $per_page): ?>
                <div style="margin-top: 1rem; text-align: center;">
                    <?php
                    $total_pages = ceil($total_quotes / $per_page);
                    for ($i = 1; $i <= $total_pages; $i++):
                    ?>
                        <a href="?page=<?= $i ?>&search=<?= urlencode($search) ?>" 
                           class="btn <?= $i === $page ? 'btn-primary' : 'btn-secondary' ?> btn-small" 
                           style="margin: 0 0.25rem;">
                            <?= $i ?>
                        </a>
                    <?php endfor; ?>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<?php require_once '../includes/footer.php'; ?>