<?php
require_once '../includes/db.php';
require_once '../includes/auth.php';
require_login();

$search = trim($_GET['search'] ?? '');
$status_filter = $_GET['status'] ?? '';
$page = max(1, intval($_GET['page'] ?? 1));
$per_page = 20;
$offset = ($page - 1) * $per_page;

try {
    $where_conditions = [];
    $params = [];
    
    if ($search) {
        $where_conditions[] = "(tracking_number LIKE ? OR customer_name LIKE ?)";
        $params[] = "%$search%";
        $params[] = "%$search%";
    }
    
    if ($status_filter) {
        $where_conditions[] = "status = ?";
        $params[] = $status_filter;
    }
    
    $where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';
    
    // Get total count
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM shipments $where_clause");
    $stmt->execute($params);
    $total_shipments = $stmt->fetch()['total'];
    
    // Get shipments
    $stmt = $pdo->prepare("
        SELECT * FROM shipments 
        $where_clause 
        ORDER BY date_created DESC 
        LIMIT $per_page OFFSET $offset
    ");
    $stmt->execute($params);
    $shipments = $stmt->fetchAll();
    
    // Get distinct statuses for filter
    $stmt = $pdo->query("SELECT DISTINCT status FROM shipments ORDER BY status");
    $statuses = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
} catch (PDOException $e) {
    error_log("Shipments page error: " . $e->getMessage());
    $shipments = [];
    $total_shipments = 0;
    $statuses = [];
}

$page_title = "Manage Shipments";
require_once '../includes/header.php';
?>

<div class="container">
    <div class="flex justify-between items-center mb-3">
        <h1>Manage Shipments</h1>
        <div style="display: flex; gap: 1rem;">
            <a href="/admin/shipment_form.php" class="btn btn-primary">Create Shipment</a>
            <a href="/admin/dashboard.php" class="btn btn-secondary">← Dashboard</a>
        </div>
    </div>
    
    <div class="card">
        <div class="search-container">
            <form method="GET" class="flex gap-2">
                <input 
                    type="text" 
                    name="search" 
                    value="<?= htmlspecialchars($search) ?>"
                    placeholder="Search by tracking number or customer name..."
                    class="form-control search-input"
                >
                <select name="status" class="form-control form-select" style="min-width: 150px;">
                    <option value="">All Status</option>
                    <?php foreach ($statuses as $status): ?>
                        <option value="<?= $status ?>" <?= $status === $status_filter ? 'selected' : '' ?>>
                            <?= htmlspecialchars($status) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <button type="submit" class="btn btn-primary">Filter</button>
                <?php if ($search || $status_filter): ?>
                    <a href="/admin/shipments.php" class="btn btn-secondary">Clear</a>
                <?php endif; ?>
            </form>
        </div>
        
        <?php if (empty($shipments)): ?>
            <div class="text-center p-2">
                <p style="color: var(--medium-gray);">
                    <?= ($search || $status_filter) ? 'No shipments found matching your criteria.' : 'No shipments created yet.' ?>
                </p>
            </div>
        <?php else: ?>
            <div class="table-container">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Tracking #</th>
                            <th>Customer</th>
                            <th>Type</th>
                            <th>Route</th>
                            <th>Status</th>
                            <th>ETA</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($shipments as $shipment): ?>
                            <tr>
                                <td><?= htmlspecialchars($shipment['tracking_number']) ?></td>
                                <td><?= htmlspecialchars($shipment['customer_name']) ?></td>
                                <td><?= htmlspecialchars($shipment['cargo_type']) ?></td>
                                <td><?= htmlspecialchars($shipment['origin']) ?> → <?= htmlspecialchars($shipment['destination']) ?></td>
                                <td>
                                    <span class="status-badge status-<?= strtolower(str_replace(' ', '', $shipment['status'])) ?>">
                                        <?= htmlspecialchars($shipment['status']) ?>
                                    </span>
                                </td>
                                <td><?= $shipment['eta'] ? date('M j, Y', strtotime($shipment['eta'])) : 'TBD' ?></td>
                                <td>
                                    <div style="display: flex; gap: 0.5rem;">
                                        <a href="/admin/shipment_form.php?id=<?= $shipment['id'] ?>" class="btn btn-small btn-secondary">Edit</a>
                                        <form method="POST" action="/admin/delete_shipment.php" style="display: inline;" onsubmit="return confirmDelete()">
                                            <?= csrf_field() ?>
                                            <input type="hidden" name="shipment_id" value="<?= $shipment['id'] ?>">
                                            <button type="submit" class="btn btn-small btn-danger">Delete</button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<?php require_once '../includes/footer.php'; ?>